<?php
/**
 * Plugin Name: YouSeeMe Payment
 * Plugin URI: https://wordpress.org/plugins/youseeme/
 * Description: Accept cryptocurrencies as payment method.
 * Author: YouSeeMe
 * Author URI: https://youseeme.io/
 * Version: 1.1.1
 * Requires at least: 6.0
 * Tested up to: 6.2.2
 * WC requires at least: 7.9.0
 * WC tested up to: 7.9.0
 * Text Domain: youseeme
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Required minimums and constants
 */
define( 'YOUSEEME_VERSION', '1.1.1' ); // WRCS: DEFINED_VERSION.
define( 'YOUSEEME_MIN_PHP_VER', '7.3.0' );
define( 'YOUSEEME_MIN_WC_VER', '7.9.0' );
define( 'YOUSEEME_FUTURE_MIN_WC_VER', '7.9.0' );
define( 'YOUSEEME_MAIN_FILE', __FILE__ );
define( 'YOUSEEME_ABSPATH', __DIR__ . '/' );
define('YOUSEEME_COMMISSION', '5');
define( 'YOUSEEME_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
define( 'YOUSEEME_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );




add_action( 'plugins_loaded', 'youseeme_init' );

function youseeme_init(): void
{
    load_plugin_textdomain( 'youseeme', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );

    if ( ! class_exists( 'WooCommerce' ) ) {
        add_action( 'admin_notices', 'youseeme_missing_wc_notice' );
        return;
    }

    if ( version_compare( WC_VERSION, YOUSEEME_MIN_WC_VER, '<' ) ) {
        add_action( 'admin_notices', 'youseeme_wc_not_supported' );
        return;
    }

    youseeme();
}


/**
 * WooCommerce fallback notice.
 *
 * @since 1.0.0
 */
function youseeme_missing_wc_notice(): void
{
    /* translators: 1. URL link. */
    echo '<div class="error"><p><strong>' . sprintf( esc_html__( 'Youseeme requires WooCommerce to be installed and active. You can download %s here.', 'youseeeme' ), '<a href="https://woocommerce.com/" target="_blank">WooCommerce</a>' ) . '</strong></p></div>';
}

/**
 * WooCommerce not supported fallback notice.
 *
 * @since 1.1.0
 */
function youseeme_wc_not_supported(): void
{
    /* translators: $1. Minimum WooCommerce version. $2. Current WooCommerce version. */
    echo '<div class="error"><p><strong>' . sprintf( esc_html__( 'Youseeme requires WooCommerce %1$s or greater to be installed and active. WooCommerce %2$s is no longer supported.', 'youseeeme' ), esc_html( YOUSEEME_MIN_WC_VER ), esc_html( WC_VERSION ) ) . '</strong></p></div>';
}



function youseeme(): void
{

    static $plugin;

    if ( ! isset( $plugin ) ) {


        class Youseeme extends WC_Settings_API
        {
            /**
             * The *Singleton* instance of this class
             *
             * @var Youseeme
             */
            private static $instance;

            /**
             * Returns the *Singleton* instance of this class.
             *
             * @return Youseeme The *Singleton* instance.
             */
            public static function get_instance() {
                if ( null === self::$instance ) {
                    self::$instance = new self();
                }
                return self::$instance;
            }

            /**
             * Protected constructor to prevent creating a new instance of the
             * *Singleton* via the `new` operator from outside of this class.
             */
            public function __construct() {
                $this->init();
            }

            public function init (): void
            {
                require_once dirname( __FILE__ ) . '/includes/class-wc-gateway-youseeme.php';
                require_once dirname( __FILE__ ) . '/includes/class-wc-gateway-youseeme-iban.php';

                add_action( 'init', [ $this,'register_youseeme_pending_order_status'] );
                add_action('add_meta_boxes_shop_order',[ $this, 'add_crypto_values_metabox']);
                add_filter( 'woocommerce_payment_gateways', [ $this, 'add_gateways' ] );
                add_filter('wc_order_statuses', array( $this, 'add_pending_crypto_status' ));

                $cryptos = get_option(
                      'youseeme_cryptocurrencies',
                      array(
                            array(
                                  'crypto_name'   => $this->get_option( 'crypto_name' ),
                                  'crypto_address' => $this->get_option( 'crypto_address' ),
                            ),
                      )
                );

                $ibans = get_option('youseeme_ibans',[
                    [
                        'bank_name' => $this->get_option( 'bank_name'),
                        'iban' => $this->get_option( 'iban'),
                        'bic' => $this->get_option( 'bic'),
                    ]
                ]);

                if(count($ibans) > 0){
                    add_filter( 'woocommerce_payment_gateways', [ $this, 'remove_BACS_gateway' ] );
                }
            }

            public function add_crypto_values_metabox (): void
            {
                add_meta_box(
                      'crypto_values',
                      'YOUSEEME Payment',
                      [$this,'crypto_values_metabox_html'],
                      'shop_order',
                      'side',
                      'high'
                );
            }

            public function crypto_values_metabox_html ($post): void
            {
                $youseeme_rate = get_post_meta($post->ID, 'youseeme_rate', true);
                $youseeme_crypto = get_post_meta($post->ID, 'youseeme_crypto', true);
                $youseeme_total = get_post_meta($post->ID, 'youseeme_total', true);
                $youseeme_iban_bank = get_post_meta($post->ID, 'youseeme_iban_bank', true);
                $youseeme_iban_total = get_post_meta($post->ID, 'youseeme_iban_total', true);
                if(!empty($youseeme_rate)){
                ?>
                <strong>Total Crypto <?= $youseeme_total; ?> <?= $youseeme_crypto; ?> (<?= number_format($youseeme_rate,15); ?>)</strong>
                <p><?= __("Please note that payment is made from your customer to your direct Wallet, wait for Youseeme to send you a confirmation notice for receipt of the amount before validating the order.","youseeme")?></p>
                <?php
                }
                else if (!empty($youseeme_iban_bank)){
                    ?>
                    <strong>IBAN: <?= $youseeme_iban_bank; ?> </strong>
                    <br>
                    <span>Total: <?= $youseeme_iban_total ?>€</span>
                    <?php
                }

            }

            public function register_youseeme_pending_order_status(): void
            {
                register_post_status( 'wc-youseeme-pending', array(
                      'label'                     => __('Awaiting Crypto Confirmation', 'youseeme'),
                      'public'                    => true,
                      'show_in_admin_status_list' => true,
                      'show_in_admin_all_list'    => true,
                      'exclude_from_search'       => false,
                      'label_count'               => _n_noop( 'Awaiting Crypto Confirmation <span class="count">(%s)</span>', 'Awaiting Crypto Confirmation <span class="count">(%s)</span>' )
                ) );
            }

            public function add_pending_crypto_status($statuses){
                $statuses['wc-youseeme-pending'] =  __('Awaiting Crypto Confirmation', 'youseeme');
                return $statuses;
            }

            public function add_gateways ($methods): array
            {
                $methods[] = WC_Gateway_Youseeme::class;
                $methods[] = WC_Gateway_Youseeme_IBAN::class;

                return $methods;
            }

            public function remove_BACS_gateway ($methods): array
            {
                return array_filter($methods,function($method){
                    return $method !== "WC_Gateway_BACS";
                });
            }
        }

        $plugin = Youseeme::get_instance();
    }
}
