
<?php
class WC_Gateway_Youseeme extends WC_Payment_Gateway{

    const ID = 'youseeme';

    public array $cryptos;

    protected string $apiKey;
    protected string $commission;
    protected array $available_cryptos = ["UNI","FTM","SHIB","USDC","dogecoin","CARDANO","BINANCE USD","SOLANA","TRON","bitcoincash","EUREC","BTC","ETH","LTC","USDT","YOUSY","WBTC","FLOWS","DASH","DAI","1INCH","BNB","MKR","SUSHI","MATIC","OMG","AUDIO","OP","ALICE","AVAX","UBN"];

    public function __construct ()
    {
        $this->id           = self::ID;
        $this->method_title = __( 'Youseeme - Cryptocurrencies', 'youseeme' );
        $this->title = __( 'Pay using Crypto', 'youseeme' );
        $this->method_description = __( 'Accept cryptocurrencies as payment method.', 'youseeme' );
        $this->has_fields         = true;
        $this->supports           = [
              'products',
              'refunds',
              'tokenization',
              'add_payment_method',
        ];

        $this->description = __( 'Pay using your favorite Cryptocurrency', 'youseeme' );
        $this->apiKey = $this->get_option('api_key') ;
        $this->commission = !empty($this->get_option('commission')) ? $this->get_option('commission') : YOUSEEME_COMMISSION;
        $this->init_form_fields();
        $this->init_settings();

        $this->cryptos = get_option(
              'youseeme_cryptocurrencies',
              array(
                    array(
                          'crypto_name'   => $this->get_option( 'crypto_name' ),
                          'crypto_address' => $this->get_option( 'crypto_address' ),
                    ),
              )
        );

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'save_cryptocurrencies' ) );
        add_action('admin_enqueue_scripts', [$this,'enqueue_admin_script'] );

    }

    public function process_admin_options ()
    {
        $post_data = $this->get_post_data();

        if(empty($post_data['woocommerce_youseeme_api_key']) || !$this->getCryptoRates($post_data['woocommerce_youseeme_api_key'])){
            //wp_die('Veuillez contacter Youseeme pour obtenir une clé valide','Clé d\'api non valide', ['response' => 400]);
            $this->add_error("Clé d'api non valide, Veuillez contacter Youseeme pour obtenir une clé valide");
            $this->display_errors();
            return;
        }

        return parent::process_admin_options();
    }
    public function init_form_fields() {
        // Configuration des champs de réglages pour la méthode de paiement
        $this->form_fields = array(
              'enabled' => array(
                    'title'   => __( 'Activate/Désactivate', 'youseeme' ),
                    'type'    => 'checkbox',
                    'label'   => __( 'Activate payment by cryptocurrencies', 'youseeme' ),
                    'default' => 'yes',
                    'required' => true,
              ),
              'api_key' => array(
                    'title'       => __( 'API key <sup style="color:red;">*</sup>', 'youseeme' ),
                    'type'        => 'text',
                    'description' => __( 'Please sign up at <a href="https://user.youseeme.io">https://user.youseeme.io</a> for the opening of your free wallet for your cash in collection, and the API key for the rates. Or <a href="mailto:contact@youseeme.pro">contact@youseeme.pro</a>', 'youseeme' ),
                    'default'     => '',
                    'desc_tip'    => false,
                    'required' => true,
              ),
              'commission' => array(
                    'title'       => __( 'Commission (%) <sup style="color:red;">*</sup>', 'youseeme' ),
                    'type'        => 'text',
                    'description' =>  '',
                    'default'     => $this->commission,
                    'desc_tip'    => false,
                    'required' => true,
              ),
              'cryptocurrencies' => array(
                    'type' => 'cryptocurrencies',
              ),
        );
    }

    public function enqueue_admin_script ()
    {
        wp_enqueue_script('youseeme-admin-crypto', plugins_url('youseeme').'/assets/admin-crypto.js', ['jquery']);
    }

    /**
     * Generate account details html.
     *
     * @return string
     */
    public function generate_cryptocurrencies_html(): string
    {
        ob_start();
        ?>
        <tr>
            <th scope="row" class="titledesc"><?php esc_html_e( 'Cryptocurrencies', 'youseeme' ); ?></th>
            <td class="forminp" id="youseeme_cryptos">
                <div class="wc_input_table_wrapper">
                    <table class="widefat wc_input_table sortable">
                        <thead>
                        <tr>
                            <th class="sort">&nbsp;</th>
                            <th><?php esc_html_e( 'Crypto', 'youseeme' ); ?></th>
                            <th><?php esc_html_e( 'Address', 'youseeme' ); ?></th>
                            <th><?php esc_html_e( 'QR code', 'woocommerce' ); ?></th>
                        </tr>
                        </thead>
                        <tbody class="accounts">
                        <?php
                        $i = -1;
                        if ( $this->cryptos ) {
                            foreach ( $this->cryptos as $crypto ) {
                                $i++;
                                echo  '<tr class="cryptos">
                                    <td class="sort"></td>
                                    <td><b>'.esc_attr( $crypto['crypto_name'] ).'</b><input id="mycryptosName_'.esc_attr( $i ).'" type="hidden" class="crypto_name" value="' . esc_attr( $crypto['crypto_name'] ) . '"  name="crypto_name[' . esc_attr( $i ) . ']" /></td>
                                    <td><input type="text" value="' . esc_attr( $crypto['crypto_address'] ) . '" class="crypto_address" placeholder="ex: 02f5e25778dcee9539b25799831277eb8e73" name="crypto_address[' . esc_attr( $i ) . ']" id="mycryptosAddress_'.esc_attr( $i ).'" onChange="updateQR()" /></td>
                                    <td><img id = "qr_' . esc_attr( $i ) .'" width="40" height="40" src="" alt="qr_code" /> </td>
                                </tr>';
                            }
                        }
                        ?>
                        </tbody>
                        <tfoot>
                        <tr>
                            <th colspan="7"><a href="#" class="add button"><?php esc_html_e( '+ Add crypto', 'youseeme' ); ?></a> <a href="#" class="remove_rows button"><?php esc_html_e( 'Remove selected cryptos(s)', 'youseeme' ); ?></a></th>
                        </tr>
                        </tfoot>
                    </table>
                </div>
            </td>
        </tr>
        <?php
        return ob_get_clean();

    }

    /**
     * Save account details table.
     */
    public function save_cryptocurrencies() {

        $cryptocurrencies = array();


        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verification already handled in WC_Admin_Settings::save()
        if ( isset( $_POST['crypto_name'] ) && isset( $_POST['crypto_address'] ) ) {

            $crypto_names   = wc_clean( wp_unslash( $_POST['crypto_name'] ) );
            $crypto_addresses = wc_clean( wp_unslash( $_POST['crypto_address'] ) );
            $crypto_address_BIC = isset($_POST['crypto_address_BIC']) ? wc_clean( wp_unslash( $_POST['crypto_address_BIC'] ) ) :null;


            foreach ( $crypto_names as $i => $name ) {
                if ( ! isset( $name ) ||  (isset( $crypto_addresses[ $i ]) && empty($crypto_addresses[ $i ]) )) {
                    continue;
                }

                $cryptocurrencies[] = array(
                      'crypto_name'   => $name,
                      'crypto_address' => $crypto_addresses[ $i ],
                );
            }
        }
        // phpcs:enable

        update_option( 'youseeme_cryptocurrencies', $cryptocurrencies );
    }


    public function payment_fields ()
    {
        $description          = $this->get_description();
        $description          = ! empty( $description ) ? $description : '';

        if(empty($this->apiKey)){
            return;
        }


        $vars = $this->getTemplateVarInfos();

        wp_enqueue_script('youseeme-front-crypto', plugins_url('youseeme').'/assets/front-crypto.js', ['jquery']);
        wp_localize_script('youseeme-front-crypto','cryptosDatas',[
            'cryptosJson'=> $vars['cryptosJson'],
              'labelAmount' =>__("Total amount to pay:","youseeme"),
              'labelAddress' =>__("Public address","youseeme"),
        ]);
        if(count($this->cryptos)){
            $html = "<div class='col2-set'>
                        <div class='col-1'>
                            <p>".$description."</p>
                            <p><select  style='padding:8px;width:250px' onchange='updatePriceAndQr()' id='cryptoSelect'>
                                <<option value='-'>".__( 'Please select your favorite Crypto' ,'youseeme')."</option>";
                                foreach ($this->cryptos as $crypto ) {
                                    $html.= "<option value='".$crypto['crypto_name']."'>".$crypto['crypto_name']."</option>";
                                }
                    $html .= '</select></p>
                            <div id="crypto_price"></div>
                            <br/>
                            <p id="crypto_amount_label" style="display:none;">'.__("Please SCAN THE QR CODE to make your payment or COPY THE ADDRESS BELOW and paste it into your wallet to complete the payment.","youseeme").'</p>
                            <div id="crypto_address"></div>
                            <input type="hidden" name="youseeme_total" value="">
                            <input type="hidden" name="youseeme_rate" value="">
                            <input type="hidden" name="youseeme_crypto" value="">
                        </div>
                        <div class="col-2">
                            <img alt="Youpay logo" style=";margin:1rem 0;" src="'.plugins_url("youseeme").'/assets/YouPay_cryptopayment.png" width="154"/>
                            <img src="" id="crypto_QR" alt="qr_code" style="display:none" />
                        </div>
                    </div>';

            echo $html;
        }
    }

    public function getTemplateVarInfos(): array
    {
        $totalAmount                = WC()->cart->total;
        $responseObject =  $this->getCryptoRates($this->apiKey);
        $vars = ['total' => $totalAmount];

        if(count($this->cryptos)>0){
            foreach ($this->cryptos as $crypto){
                if($crypto["crypto_name"] === "IBAN"){
                    $mytotal = (float) $totalAmount;
                    $rate=1;
                }
                else {
                    $cryptoEur = $crypto['crypto_name'] . '/EUR';
                    $rate = $responseObject->$cryptoEur;
                    $mytotal = ((float) $totalAmount / (float) $rate) * (1 + ($this->commission / 100));
                }

                $vars['cryptos'][] = ['name' => $crypto['crypto_name'], 'total' => $mytotal, 'key' => $crypto['crypto_address'],'rate' => (float) $rate ];
            }
        }

        $vars['cryptosJson'] = json_encode($vars['cryptos']);
        return $vars;
    }

    public function getCryptoRates ($apiKey)
    {
        $hashed = hash('sha512', $apiKey . '-' . get_site_url());
        $responseObject = wp_remote_get('https://api.youseeme.io/v1/ecommerce/plugins/crypto/prices', [
            'headers' => array(
                  'Content-Type' => 'application/json',
                  'X-Api-Key' => $apiKey,
                  'serial-id '=> $hashed,
            )
        ]);
        if(!isset($responseObject['response']) || $responseObject['response']['code'] !== 200){
            return false;
        }

        return json_decode($responseObject['body']);
    }

    public function process_payment($order_id) {
        $order = wc_get_order( $order_id );

        try{
            if($_POST['payment_method'] === 'youseeme'
                  && isset($_POST['youseeme_total']) && !empty($_POST['youseeme_total'])
                  && isset($_POST['youseeme_rate']) && !empty($_POST['youseeme_rate'])
                  && isset($_POST['youseeme_crypto']) && !empty($_POST['youseeme_crypto'])
            ){
                $order->update_status('wc-youseeme-pending');
                $order->update_meta_data( 'youseeme_total', floatval($_POST['youseeme_total']) );
                $order->update_meta_data( 'youseeme_rate', floatval($_POST['youseeme_rate']) );

                if(!in_array($_POST['youseeme_crypto'], $this->available_cryptos)){
                    throw new Exception("Unavailable crypto name");
                }
                $order->update_meta_data( 'youseeme_crypto', $_POST['youseeme_crypto'] );
                $order->save();
                WC()->cart->empty_cart();

                return array(
                      'result'   => 'success',
                      'redirect' => $this->get_return_url($order),
                );
            }
        } catch ( Exception $e ) {
            wc_add_notice( $e->getMessage(), 'error' );

            $statuses = [ 'pending', 'failed' ];

            if ( $order->has_status( $statuses ) ) {
                $this->send_failed_order_email( $order_id );
            }

            return [
                  'result'   => 'fail',
                  'redirect' => '',
            ];
        }
    }

    /**
     * Sends the failed order email to admin.
     *
     * @param int $order_id
     * @return null
     *@since 1.0.0
     * @version 1.0.0
     */
    public function send_failed_order_email( int $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_Failed_Order']->trigger( $order_id );
        }
    }


    /**
     * Returns all supported currencies for this payment method.
     *
     * @since 1.0.0
     * @version 1.0.0
     * @return array
     */
    public function get_supported_currency(): array
    {
        return apply_filters(
              'youseeme_supported_currencies',
              [
                    'EUR',
              ]
        );
    }

    /**
     * Checks to see if all criteria is met before showing payment method.
     *
     * @since 1.0.0
     * @version 1.0.0
     * @return bool
     */
    public function is_available(): bool
    {

        if ( ! in_array( get_woocommerce_currency(), $this->get_supported_currency() ) ) {
            return false;
        }

        return parent::is_available();
    }


}
