
<?php
class WC_Gateway_Youseeme_IBAN extends WC_Payment_Gateway{

    const ID = 'youseeme-iban';

    public array $ibans;

    public function __construct ()
    {
        $this->id           = self::ID;
        $this->method_title = __( 'Youseeme - IBAN', 'youseeme' );
        $this->title = __( 'Pay using IBAN', 'youseeme' );
        $this->method_description = __( 'Accept IBAN as payment method.</br>Please subscribe to a youseeme IBAN account on <a href="https://www.youseeme.fr">https://www.youseeme.fr</a> dedicated to your online sales, it contains a Mastercard, so you will receive an essential notification of payment for each sale before validating the order, because remember there are no payment intermediaries.', 'youseeme' );
        $this->has_fields         = true;
        $this->supports           = [
              'products',
              'refunds',
              'tokenization',
              'add_payment_method',
        ];

        $this->description = __( 'Pay using bank transfer', 'youseeme' );
        $this->init_form_fields();
        $this->init_settings();

        $this->ibans = get_option(
              'youseeme_ibans',
              array(
                    array(
                          'bank_name'   => $this->get_option( 'bank_name' ),
                          'iban' => $this->get_option( 'iban' ),
                          'bic' => $this->get_option( 'bic' ),
                    ),
              )
        );

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'save_ibans' ) );
        add_action('admin_enqueue_scripts', [$this,'enqueue_admin_script'] );
    }
    public function init_form_fields() {
        // Configuration des champs de réglages pour la méthode de paiement
        $this->form_fields = array(
              'enabled' => array(
                    'title'   => __( 'Activate/Désactivate', 'youseeme' ),
                    'type'    => 'checkbox',
                    'label'   => __( 'Activate payment by iban', 'youseeme' ),
                    'default' => 'yes',
                    'required' => true,
              ),
              'ibans' => array(
                    'type' => 'ibans',
              ),
        );
    }
    public function enqueue_admin_script ()
    {
        wp_enqueue_script('youseeme-admin-ibans', plugins_url('youseeme').'/assets/admin-iban.js', ['jquery']);
    }

    /**
     * Generate account details html.
     *
     * @return string
     */
    public function generate_ibans_html(): string
    {
        ob_start();
        ?>
        <tr>
            <th scope="row" class="titledesc"><?php esc_html_e( 'Ibans', 'youseeme' ); ?></th>
            <td class="forminp" id="youseeme_ibans">
                <div class="wc_input_table_wrapper">
                    <table class="widefat wc_input_table sortable">
                        <thead>
                        <tr>
                            <th class="sort">&nbsp;</th>
                            <th><?php esc_html_e( 'Bank name', 'youseeme' ); ?></th>
                            <th><?php esc_html_e( 'IBAN', 'youseeme' ); ?></th>
                            <th><?php esc_html_e( 'BIC / SWIFT', 'youseeme' ); ?></th>
                            <th><?php esc_html_e( 'QR code', 'youseeme' ); ?></th>
                        </tr>
                        </thead>
                        <tbody class="accounts">
                        <?php
                        $i = -1;
                        if ( $this->ibans ) {
                            foreach ( $this->ibans as $iban ) {
                                $i++;
                                if( !empty($iban['iban'])){
                                    echo  '<tr class="ibans">
                                    <td class="sort"></td>
                                    <td><input id="myibansName_'.esc_attr( $i ).'" type="text" '.($iban["bank_name"] === "Youseeme"? "readonly" :"").' class="bank_name" value="' . esc_attr( $iban['bank_name'] ) . '"  name="bank_name[' . esc_attr( $i ) . ']" onChange="updateIbanQR()" /></td>
                                    <td><input type="text" value="' .esc_attr( $iban['iban'] ) . '" class="iban" placeholder="IBAN" name="iban[' . esc_attr( $i ) . ']" id="myibansAddress_'.esc_attr( $i ).'" onChange="updateIbanQR()" /></td>
                                    <td><input type="text" value="' .esc_attr( $iban['bic'] ) . '" class="bic" placeholder="BIC" name="bic[' . esc_attr( $i ) . ']" id="myibansAddress_BIC_'.esc_attr( $i ).'" onChange="updateIbanQR()" /></td>
                                    <td> <img id = "qr_' . esc_attr( $i ) .'" width="40" height="40" src="" alt="qr_code" /></td>
                                </tr>';
                                }
                            }
                        }
                        ?>
                        </tbody>
                        <tfoot>
                        <tr>
                            <th colspan="7"><a href="#" class="add button"><?php esc_html_e( '+ Add iban', 'youseeme' ); ?></a><a href="#" class="add_youseeme button"><?php esc_html_e( '+ Add Youseeme IBAN', 'youseeme' ); ?></a> <a href="#" class="remove_rows button"><?php esc_html_e( 'Remove selected ibans(s)', 'youseeme' ); ?></a></th>
                        </tr>
                        </tfoot>
                    </table>
                </div>
            </td>
        </tr>
        <?php
        return ob_get_clean();

    }

    /**
     * Save account details table.
     */
    public function save_ibans() {

        $ibansObj = array();


        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verification already handled in WC_Admin_Settings::save()
        if ( isset( $_POST['bank_name'] ) && isset( $_POST['iban'] ) ) {

            $bank_names   = wc_clean( wp_unslash( $_POST['bank_name'] ) );
            $ibans = wc_clean( wp_unslash( $_POST['iban'] ) );
            $bics = isset($_POST['bic']) ? wc_clean( wp_unslash( $_POST['bic'] ) ) :null;


            foreach ( $bank_names as $i => $name ) {
                if ( ! isset( $name ) ||  (isset( $ibans[ $i ]) && empty($ibans[ $i ]) )) {
                    continue;
                }

                $ibansObj[] = array(
                      'bank_name'   => $name,
                      'iban' => $ibans[ $i ],
                    'bic' =>$bics ? $bics[$i] : ''
                );
            }
        }
        // phpcs:enable

        update_option( 'youseeme_ibans', $ibansObj );
    }

    public function getTemplateVarInfos(): array
    {
        $totalAmount                = WC()->cart->total;
        $vars = ['total' => $totalAmount];

        if(count($this->ibans)>0){
            foreach ($this->ibans as $iban){
                $vars['ibans'][] = ['name' => $iban['bank_name'], 'total' => (float) $totalAmount, 'key' => $iban['iban'],'bic' => $iban['bic'],'rate' => 1.0 ];
            }
        }

        $vars['ibansJson'] = json_encode($vars['ibans']);
        return $vars;
    }

    public function payment_fields ()
    {
        $description          = $this->get_description();
        $description          = ! empty( $description ) ? $description : '';



        $vars = $this->getTemplateVarInfos();
        wp_enqueue_script('youseeme-front-iban', plugins_url('youseeme').'/assets/front-iban.js', ['jquery']);
        wp_localize_script('youseeme-front-iban','ibansDatas',[
              'ibansJson'=> $vars['ibansJson'],
              'labelAmount' =>__("Total amount to pay:","youseeme"),
        ]);
        if(count($this->ibans)){
            $html = "<div class='col2-set'>
                        <div class='col-1'>
                            <p>".$description."</p>
                            <p><select  style='padding:8px;width:250px' onchange='updateIbanPriceAndQr()' id='ibanSelect'>
                                <option value='-'>".__( 'Please select an IBAN account' ,'youseeme')."</option>";
                                foreach ($this->ibans as $iban ) {
                                    $html.= "<option value='".$iban['bank_name']."'>".$iban['bank_name']."</option>";
                                }
                        $html .= '</select></p>
                            <div id="iban_price"></div>
                            <br/>
                            <p id="iban_amount_label" style="display:none;">'.__("You will need to transfer the total amount of your purchase to our account. You will receive your order confirmation via email, which will include our bank details and the order number.","youseeme").'</p>
                            <div id="iban"></div>
                            <div id="bic"></div>
                            <input type="hidden" name="youseeme_iban_total" value="">
                            <input type="hidden" name="youseeme_iban_bank" value="">
                        </div>
                        <div class="col-2">
                            <img alt="Youpay logo" style=";margin:1rem 0;" src="'.plugins_url("youseeme").'/assets/YouPay_cryptopayment.png" width="154"/>
                            <img src="" id="iban_QR" alt="qr_code" style="display:none" />
                        </div>
                    </div>
               ';

            echo $html;
        }
    }


    public function process_payment($order_id) {
        $order = wc_get_order( $order_id );

        try{
            if($_POST['payment_method'] === 'youseeme-iban'
                  && isset($_POST['youseeme_iban_total']) && !empty($_POST['youseeme_iban_total'])
                  && isset($_POST['youseeme_iban_bank']) && !empty($_POST['youseeme_iban_bank'])
            ){
                $order->update_status('wc-pending');
                $order->update_meta_data( 'youseeme_iban_total', floatval($_POST['youseeme_iban_total']) );

                /*if(!in_array($_POST['youseeme_iban'], $this->available_ibans)){
                    throw new Exception("Unavailable iban name");
                }*/
                $order->update_meta_data( 'youseeme_iban_bank', $_POST['youseeme_iban_bank'] );
                $order->save();
                WC()->cart->empty_cart();

                return array(
                      'result'   => 'success',
                      'redirect' => $this->get_return_url($order),
                );
            }
        } catch ( Exception $e ) {
            wc_add_notice( $e->getMessage(), 'error' );

            $statuses = [ 'pending', 'failed' ];

            if ( $order->has_status( $statuses ) ) {
                $this->send_failed_order_email( $order_id );
            }

            return [
                  'result'   => 'fail',
                  'redirect' => '',
            ];
        }
    }

    /**
     * Sends the failed order email to admin.
     *
     * @param int $order_id
     * @return null
     *@since 1.0.0
     * @version 1.0.0
     */
    public function send_failed_order_email( int $order_id ) {
        $emails = WC()->mailer()->get_emails();
        if ( ! empty( $emails ) && ! empty( $order_id ) ) {
            $emails['WC_Email_Failed_Order']->trigger( $order_id );
        }
    }


    /**
     * Returns all supported currencies for this payment method.
     *
     * @since 1.0.0
     * @version 1.0.0
     * @return array
     */
    public function get_supported_currency(): array
    {
        return apply_filters(
              'youseeme_supported_currencies',
              [
                    'EUR',
              ]
        );
    }

    /**
     * Checks to see if all criteria is met before showing payment method.
     *
     * @since 1.0.0
     * @version 1.0.0
     * @return bool
     */
    public function is_available(): bool
    {

        if ( ! in_array( get_woocommerce_currency(), $this->get_supported_currency() ) ) {
            return false;
        }

        return parent::is_available();
    }


}
